//
// TinyPTC by Gaffer
// www.gaffer.org/tinyptc
//

#include "stdafx.h"
#include "tinyptc.h"
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>

#define WIDTH 640
#define HEIGHT 480
#define SIZE WIDTH*HEIGHT

typedef struct {
	unsigned char b;
	unsigned char g;
	unsigned char r;
	unsigned char a;
} rgb;

typedef struct {
	unsigned short h;       // angle in degrees
	unsigned char s;
	unsigned char v;
} hsv;


#define HSV(ph, ps, pv)	((hsv) { .h = ph, .s = ps, .v = pv })
#define RGB(r, g, b, a)	((rgb) { .r = r, .g = g, .b = b, .a = a })

union PIXEL
{
	rgb colors;
	int color;
};

int noise;
int carry;
unsigned int x, y, position;
int seed = 0x12345;
union PIXEL pixel[SIZE];
int plasma[SIZE];
rgb palette[256];

static hsv   rgb2hsv(rgb in);
static rgb   hsv2rgb(hsv in);

hsv rgb2hsv(rgb in)
{
	hsv         out;
	unsigned char      min, max, delta;

	min = in.r < in.g ? in.r : in.g;
	min = min  < in.b ? min : in.b;

	max = in.r > in.g ? in.r : in.g;
	max = max > in.b ? max : in.b;

	out.v = max;                                // v
	delta = max - min;
	if (delta == 0)
	{
		out.s = 0;
		out.h = 0; // undefined, maybe nan?
		return out;
	}
	if (max > 0) { // NOTE: if Max is == 0, this divide would cause a crash
		out.s = (delta / max);                  // s
	}
	else {
		// if max is 0, then r = g = b = 0
		// s = 0, h is undefined
		out.s = 0;
		out.h = 0xffff;                            // its now undefined
		return out;
	}
	if (in.r >= max)                           // > is bogus, just keeps compilor happy
		out.h = (in.g - in.b) / delta;        // between yellow & magenta
	else
		if (in.g >= max)
			out.h = 2*255 + (in.b - in.r) / delta;  // between cyan & yellow
		else
			out.h = 4*255 + (in.r - in.g) / delta;  // between magenta & cyan

	out.h *= 60;                              // degrees

	if (out.h < 0)
		out.h += 360;

	return out;
}

rgb hsv2rgb(hsv in)
{
	unsigned short      hh;
	unsigned short       p, q, t, ff;
	long        i;
	rgb         out;

	if (in.s <= 0) {       // < is bogus, just shuts up warnings
		out.r = in.v;
		out.g = in.v;
		out.b = in.v;
		return out;
	}
	hh = in.h;
	if (hh >= 360) hh = 0;
	hh /= 60;
	i = (long)hh;
	ff = hh - i;
	p = in.v * (255 - in.s);
	q = in.v * (255 - (in.s * ff));
	t = in.v * (255 - (in.s * (255 - ff)));

	switch (i) {
	case 0:
		out.r = in.v;
		out.g = t;
		out.b = p;
		break;
	case 1:
		out.r = q;
		out.g = in.v;
		out.b = p;
		break;
	case 2:
		out.r = p;
		out.g = in.v;
		out.b = t;
		break;

	case 3:
		out.r = p;
		out.g = q;
		out.b = in.v;
		break;
	case 4:
		out.r = t;
		out.g = p;
		out.b = in.v;
		break;
	case 5:
	default:
		out.r = in.v;
		out.g = p;
		out.b = q;
		break;
	}
	return out;
}

void generateRand()
{
	for (x = 0; x < SIZE; x++)
	{
		noise = seed;
		noise >>= 3;
		noise ^= seed;
		carry = noise & 1;
		noise >>= 1;
		seed >>= 1;
		seed |= (carry << 30);
		noise &= 0xFF;
		pixel[x].color = (noise << 16) | (noise << 8) | noise;
	}
}

void generateRandOneLine()
{
	for (x = 1; x < WIDTH; x++)
	{
		if (rand() % 2)
			pixel[x].color = 0xff;
		else
			pixel[x].color = 0;
	}
}

void generateMire()
{
	for (y = 0; y < HEIGHT; y++)
	{
		for (x = 0; x < WIDTH; x++)
		{
			position = (y*WIDTH) + x;
			pixel[position].color = 0xf << (x % 32);
		}
	}
}

void generatePlasma()
{
	for (y = 0; y < HEIGHT; y++)
	{
		for (x = 0; x < WIDTH; x++)
		{
			//noise = (int)(128.0 + (128.0 * sin(sqrt((x - WIDTH / 2.0) * (x - WIDTH / 2.0) + (y - HEIGHT / 2.0) * (y - HEIGHT / 2.0)) / 8.0)));
			//noise = (int)(128.0 + (128.0 * sin(x / 8.0)) + 128.0 + (128.0 * sin(y / 8.0))) / 2.0;
			position = (y*WIDTH) + x;
			pixel[position].color = (noise << 16) | (noise << 8) | noise;
		}
	}
}

int main()
{
	/*union PIXEL pouet1;
	int pouet2;
	printf("%d %d", sizeof(pouet1), sizeof(pouet2));
	return 0;*/

	time_t t0 = time(NULL);
	srand((unsigned int)t0);
	if (!ptc_open("test", WIDTH, HEIGHT)) return 1;

	rgb colorRGB;
	for (int x = 0; x < 256; x++)
	{
		//use HSVtoRGB to vary the Hue of the color through the palette
		colorRGB = hsv2rgb(HSV(x, 255, 255));
		palette[x] = colorRGB;
	}
	for (y = 0; y < HEIGHT; y++)
	{
		for (x = 0; x < WIDTH; x++)
		{
			noise = (int)((128.0 + (128.0 * sin(x / 16.0))
				+ 128.0 + (128.0 * sin(y / 16.0))
				) / 2.0);
			position = (y*WIDTH) + x;
			plasma[position] = noise;
		}
	}

	int paletteShift;

	while (1)
	{
		paletteShift = (time(NULL) - t0) / 10.0;
		for (y = 0; y < HEIGHT; y++)
		{
			for (x = 0; x < WIDTH; x++)
			{
				//buffer[y][x] = palette[(plasma[y][x] + paletteShift) % 256];
				position = (y*WIDTH) + x;
				pixel[position].colors = palette[(plasma[position] + paletteShift) % 256];
			}
		}
		//generatePlasma();
		ptc_update(pixel);
	}
}
